/*:
 * @target MZ
 * @plugindesc ADVバックログ＋ボイス再生 v1.6.0（CoreSafe+Lock）— SimpleVoiceの次メッセージだけに確実連動・入力ロック・最前面Overlay
 * @author Human
 * @help
 * ■今回の修正（v1.6.0）
 * - SimpleVoice の PLAY_VOICE 実行後、**直後の「文章の表示」1回**だけにボイスを結びつける
 *   「予約スロット」方式に変更（以降は自動で予約消費＆クリア）。
 * - バックログのリプレイ再生は**予約に影響しない**ようガード（誤連動防止）。
 * - 既存の機能（入力ロック・Interpreter停止・最前面表示・スイッチ制御）は維持。
 *
 * ■使い方
 * 1) SimpleVoice.js の「下」に本プラグインを配置・有効化（過去版はOFF推奨）。
 * 2) イベントは「SimpleVoice: PLAY_VOICE → 直後に『文章の表示』」でOK。
 * 3) バックログはプラグインコマンド OpenBacklog / ForceOpenBacklog、または任意ホットキーで開く。
 *
 * @param openMode
 * @text オープンモード
 * @type select
 * @option Overlay（シーン切替なし・最前面） @value overlay
 * @option Scene（別シーンで開く）           @value scene
 * @default overlay
 *
 * @param maxEntries
 * @text 最大記録数
 * @type number
 * @min 50
 * @max 2000
 * @default 300
 *
 * @param hotkeySymbol
 * @text ホットキー（none=無効）
 * @type select
 * @option none
 * @option pageup
 * @option pagedown
 * @option shift
 * @option control
 * @option tab
 * @option cancel
 * @option menu
 * @default none
 *
 * @param showNameInList
 * @text 名前を先頭に表示
 * @type boolean
 * @on はい
 * @off いいえ
 * @default true
 *
 * @param overlayDimOpacity
 * @text オーバーレイ暗幕（0-255）
 * @type number
 * @min 0
 * @max 255
 * @default 96
 *
 * @param enableSwitchId
 * @text 有効スイッチ（0=無効）
 * @type switch
 * @default 0
 *
 * @param autoOpenSwitchId
 * @text オートオープンスイッチ（0=無効）
 * @type switch
 * @default 0
 *
 * @param lockTouchDuringOpen
 * @text 開いている間はタッチ無効
 * @type boolean
 * @on はい
 * @off いいえ
 * @default true
 *
 * @param lockInterpreterDuringOpen
 * @text 開いている間はイベント停止
 * @type boolean
 * @on はい
 * @off いいえ
 * @default true
 *
 * @command OpenBacklog
 * @text バックログを開く
 * @desc 有効スイッチがONのときに開きます。
 *
 * @command ForceOpenBacklog
 * @text バックログを強制で開く
 * @desc 有効スイッチに関係なく開きます（デバッグ用）。
 *
 * @command ClearBacklog
 * @text バックログを消去
 * @desc 現在のバックログをすべて消去します。
 */
(() => {
  'use strict';

  //================ パラメータ =================
  const script = document.currentScript || (function(){const s=document.getElementsByTagName('script');return s[s.length-1];})();
  const matched = (script && script.src) ? script.src.match(/([^\/]+)\.js$/i) : null;
  const PLUGIN_NAME = matched ? matched[1] : 'HS_BacklogVoice_CoreSafeLock';

  const P = PluginManager.parameters(PLUGIN_NAME);
  const OPEN_MODE   = String(P.openMode || 'overlay');
  const MAX_ENTRIES = Number(P.maxEntries || 300);
  const HOTKEY      = String(P.hotkeySymbol || 'none');
  const SHOW_NAME   = P.showNameInList === 'true';
  const OVERLAY_DIM = Number(P.overlayDimOpacity || 96);
  const ENABLE_SW   = Number(P.enableSwitchId || 0);
  const AUTO_SW     = Number(P.autoOpenSwitchId || 0);
  const LOCK_TOUCH  = P.lockTouchDuringOpen === 'true';
  const LOCK_INT    = P.lockInterpreterDuringOpen === 'true';

  const safe = (fn,fb)=>{ try { return fn(); } catch(_) { return fb; } };
  const allowOpen = ()=> ENABLE_SW===0 || safe(()=> $gameSwitches.value(ENABLE_SW), true);

  //================ 「次のShow Textだけ」用 予約スロット =================
  const HS = {
    voiceReservation: null, // {voice:{name,volume,pitch,pan}, channel:number, loop:boolean}
    backlogPlayGuard: false, // バックログの再生時は予約を立てない
    setReservationFromSimpleVoice(v, loop, ch){
      this.voiceReservation = {
        voice: { name:String(v?.name||''), volume:Number(v?.volume??100), pitch:Number(v?.pitch??100), pan:Number(v?.pan??0) },
        channel: Number(ch??0),
        loop: !!loop
      };
    },
    consumeReservation(){ const r=this.voiceReservation; this.voiceReservation=null; return r; }
  };

  //================ SimpleVoice 連動（予約式） =================
  // SimpleVoice は AudioManager.playVoice(voice, loop, channel) を呼ぶ
  // → その呼び出しをフックして「予約」を立てる（ただしバックログ再生時は除外）
  if (AudioManager.playVoice) {
    const _playVoice = AudioManager.playVoice;
    AudioManager.playVoice = function(voice, loop, channel){
      const res = _playVoice.apply(this, arguments);
      try {
        if (!HS.backlogPlayGuard && voice && voice.name != null) {
          HS.setReservationFromSimpleVoice(voice, loop, channel);
        }
      } catch(_) {}
      return res;
    };
  } else {
    // SimpleVoice未導入の保険（SE扱い）
    const _playSe = AudioManager.playSe;
    AudioManager.playSe = function(se){
      const res = _playSe.apply(this, arguments);
      try {
        if (!HS.backlogPlayGuard && se && se.name != null) {
          HS.setReservationFromSimpleVoice({name:se.name, volume:se.volume??90, pitch:se.pitch??100, pan:se.pan??0}, false, 0);
        }
      } catch(_) {}
      return res;
    };
  }

  //================ 文字列の軽量展開 =================
  function conv(text){
    if (!text) return '';
    try {
      text = text.replace(/\\V\[(\d+)\]/gi, (_,n)=> String($gameVariables.value(Number(n)) ?? 0));
      text = text.replace(/\\N\[(\d+)\]/gi, (_,n)=> { const a=$gameActors.actor(Number(n)); return a? a.name() : ''; });
      text = text.replace(/\\P\[(\d+)\]/gi, (_,n)=> { const m=$gameParty.members()[Number(n)-1]; return m? m.name() : ''; });
      text = text
        .replace(/\\C\[\d+\]/gi,'').replace(/\\I\[\d+\]/gi,'').replace(/\\FS\[\d+\]/gi,'')
        .replace(/\\\\/g,'\\')
        .replace(/\\\{|\\\}|\\\.|\\\||\\\!|\\\>|\\\<|\\\^|\\\#|\\G|\\W\[\d+\]/gi,'');
      return text;
    } catch(_){ return String(text); }
  }

  //================ データ保持 =================
  const _Game_System_initialize = Game_System.prototype.initialize;
  Game_System.prototype.initialize = function(){
    _Game_System_initialize.apply(this, arguments);
    this._hsBacklog = [];
  };
  Game_System.prototype.hsPushBacklog = function(e){
    this._hsBacklog.push(e);
    if (this._hsBacklog.length > MAX_ENTRIES) this._hsBacklog.shift();
  };
  Game_System.prototype.hsBacklog = function(){ return this._hsBacklog; };
  Game_System.prototype.hsClearBacklog = function(){ this._hsBacklog = []; };

  //================ $gameMessage だけ監視（描画非干渉） =================
  const Buf = {
    active:false, lines:[], speaker:'', faceName:'', faceIndex:0,
    voice:null, // {voice:{..}, channel, loop}
    reset(){ this.active=false; this.lines.length=0; this.speaker=''; this.faceName=''; this.faceIndex=0; this.voice=null; }
  };

  const _GM_add = Game_Message.prototype.add;
  Game_Message.prototype.add = function(text){
    _GM_add.apply(this, arguments);
    try {
      if (!Buf.active){
        Buf.active = true;
        Buf.speaker   = safe(()=>this.speakerName(), this._speakerName || '') || '';
        Buf.faceName  = safe(()=>this.faceName(), this._faceName || '') || '';
        Buf.faceIndex = Number(safe(()=>this.faceIndex(), this._faceIndex || 0)) || 0;
        // ★ ここで“予約”を消費：このメッセージだけに結びつける
        Buf.voice = HS.consumeReservation(); // 使ったら即クリア
      }
      Buf.lines.push(String(text));
    } catch(_){}
  };

  const _GM_setSpeakerName = Game_Message.prototype.setSpeakerName;
  Game_Message.prototype.setSpeakerName = function(name){
    _GM_setSpeakerName.apply(this, arguments);
    try { if (Buf.active) Buf.speaker = name || Buf.speaker || ''; } catch(_){}
  };

  const _GM_setFaceImage = Game_Message.prototype.setFaceImage;
  Game_Message.prototype.setFaceImage = function(faceName, faceIndex){
    _GM_setFaceImage.apply(this, arguments);
    try { if (Buf.active){ Buf.faceName = faceName || Buf.faceName || ''; Buf.faceIndex = Number(faceIndex)||0; } } catch(_){}
  };

  const _GM_clear = Game_Message.prototype.clear;
  Game_Message.prototype.clear = function(){
    try {
      if (Buf.active){
        const raw = Buf.lines.join('\n');
        const text = conv(raw);
        if (text && text.trim().length){
          const entry = {
            speaker  : Buf.speaker || '',
            text     : text,
            faceName : Buf.faceName || '',
            faceIndex: Number.isFinite(Buf.faceIndex)? Buf.faceIndex : 0,
            voicePath: Buf.voice?.voice?.name || null,
            voiceCh  : Buf.voice?.channel ?? 0,
            voiceObj : Buf.voice?.voice || null,
            voiceLoop: !!(Buf.voice?.loop)
          };
          $gameSystem.hsPushBacklog(entry);
        }
      }
    } catch(e){ console.warn('[HS_Backlog] commit error:', e); }
    Buf.reset();
    _GM_clear.apply(this, arguments);
  };

  //================ Backlog Window =================
  function Window_HS_BacklogList(){ this.initialize(...arguments); }
  Window_HS_BacklogList.prototype = Object.create(Window_Selectable.prototype);
  Window_HS_BacklogList.prototype.constructor = Window_HS_BacklogList;

  Window_HS_BacklogList.prototype.initialize = function(rect){
    Window_Selectable.prototype.initialize.call(this, rect);
    this._data = $gameSystem.hsBacklog();
    this.activate();
    this.select(Math.max(this._data.length-1, 0));
    this.openness = 255;
  };
  Window_HS_BacklogList.prototype.maxItems = function(){ return this._data.length; };
  Window_HS_BacklogList.prototype.item = function(i){ return this._data[i]; };
  Window_HS_BacklogList.prototype.itemHeight = function(){ return this.lineHeight()*3; };
  Window_HS_BacklogList.prototype.drawItem = function(i){
    const r = this.itemLineRect(i), e = this.item(i); if (!e) return;
    const name = (SHOW_NAME && e.speaker) ? `【${e.speaker}】` : '';
    const txt  = name ? `${name} ${e.text}` : e.text;
    this.resetFontSettings();
    this.drawTextEx(txt, r.x, r.y, r.width);
    if (e.voicePath){
      this.changeTextColor(ColorManager.systemColor());
      this.drawText('●', r.x + r.width - this.textWidth('●') - 8, r.y, 32, 'right');
      this.resetTextColor();
    }
  };

  // ● 決定でボイス再生（SimpleVoiceのパラメータで再生／予約に影響させない）
  Window_HS_BacklogList.prototype.playEntryVoice = function(){
    const e = this.item(this.index()); if (!e || !e.voicePath) return;
    try{
      HS.backlogPlayGuard = true; // ← これで予約を立てない
      const v = e.voiceObj || { name:e.voicePath, volume:100, pitch:100, pan:0 };
      if (AudioManager.playVoice) {
        AudioManager.playVoice({ name:v.name, volume:v.volume, pitch:v.pitch, pan:v.pan }, !!e.voiceLoop, Number(e.voiceCh||0));
      } else {
        AudioManager.playSe({ name:e.voicePath, volume:90, pitch:100, pan:0 });
      }
    }catch(err){
      console.warn('[HS_Backlog] voice play failed:', err);
    } finally {
      HS.backlogPlayGuard = false;
    }
  };
  Window_HS_BacklogList.prototype.processOk = function(){
    Window_Selectable.prototype.processOk.call(this);
    this.playEntryVoice();
  };

  //================ Scene版（必要なら） =================
  function Scene_HS_Backlog(){ this.initialize(...arguments); }
  Scene_HS_Backlog.prototype = Object.create(Scene_MenuBase.prototype);
  Scene_HS_Backlog.prototype.constructor = Scene_HS_Backlog;
  Scene_HS_Backlog.prototype.create = function(){
    Scene_MenuBase.prototype.create.call(this);
    const rect = this.backlogRect();
    const list = new Window_HS_BacklogList(rect);
    list.setHandler('cancel', this.popScene.bind(this));
    this.addWindow(list);
    const h = new Window_Help(new Rectangle(rect.x, 0, rect.width, this.calcWindowHeight(2, false)));
    h.setText('↑↓選択 / 決定: ボイス再生 / 取消: 閉じる');
    this.addWindow(h);
  };
  Scene_HS_Backlog.prototype.backlogRect = function(){
    const wy = this.calcWindowHeight(2,false);
    return new Rectangle(0, wy, Graphics.boxWidth, Graphics.boxHeight - wy);
  };

  //================ Overlay（Scene最前面＋入力ロック） =================
  const Overlay = {
    _open:false,_list:null,_help:null,_dim:null,_swallowTouch:false,_deactMsg:false,_lockApplied:false,_origUpdate:null,
    open(){
      if (this._open) return;
      const sc = SceneManager._scene;

      const dim = new Sprite(); dim.bitmap=new Bitmap(Graphics.width,Graphics.height);
      dim.bitmap.fillRect(0,0,Graphics.width,Graphics.height,'#000'); dim.opacity = OVERLAY_DIM;
      sc.addChild(dim); this._dim = dim;

      const helpH = sc.calcWindowHeight ? sc.calcWindowHeight(2,false) : 96;
      const help = new Window_Help(new Rectangle(0,0,Graphics.boxWidth,helpH));
      help.setText('↑↓選択 / 決定: ボイス再生 / 取消: 閉じる'); sc.addChild(help); this._help = help;

      const rect = new Rectangle(0, helpH, Graphics.boxWidth, Graphics.boxHeight - helpH);
      const list = new Window_HS_BacklogList(rect); list.setHandler('cancel', ()=>Overlay.close());
      sc.addChild(list); this._list = list; list.activate(); list.select(list.maxItems()-1);

      Input.clear(); TouchInput.clear();

      const msgWin = sc._messageWindow;
      if (msgWin && msgWin.active) { msgWin.deactivate(); this._deactMsg = true; }

      if (LOCK_TOUCH) this._installTouchSwallow();
      if (LOCK_INT)   this._installInterpreterLock();

      this._open = true;
    },
    close(){
      if (!this._open) return;
      const sc = SceneManager._scene;
      if (this._list){ sc.removeChild(this._list); this._list.close(); this._list=null; }
      if (this._help){ sc.removeChild(this._help); this._help.close(); this._help=null; }
      if (this._dim){  sc.removeChild(this._dim);  this._dim.destroy({children:true}); this._dim=null; }
      if (this._deactMsg){ const msgWin = sc._messageWindow; if (msgWin) msgWin.activate(); this._deactMsg=false; }
      if (this._swallowTouch) this._uninstallTouchSwallow();
      if (this._lockApplied)  this._uninstallInterpreterLock();
      Input.clear(); TouchInput.clear();
      this._open=false;
    },
    isOpen(){ return !!this._open; },

    // タッチ無効化
    _installTouchSwallow(){
      if (this._swallowTouch) return;
      this._swallowTouch = true;
      const TI = TouchInput;
      this._origTI = {
        isTriggered: TI.isTriggered.bind(TI),
        isPressed  : TI.isPressed.bind(TI),
        isReleased : TI.isReleased ? TI.isReleased.bind(TI) : null,
        isCancelled: TI.isCancelled.bind(TI),
        wheelX     : TI.wheelX ? TI.wheelX.bind(TI) : null,
        wheelY     : TI.wheelY ? TI.wheelY.bind(TI) : null
      };
      TI.isTriggered = ()=> false;
      TI.isPressed   = ()=> false;
      TI.isCancelled = ()=> false;
      if (TI.isReleased) TI.isReleased = ()=> false;
      if (TI.wheelX) TI.wheelX = ()=> 0;
      if (TI.wheelY) TI.wheelY = ()=> 0;
    },
    _uninstallTouchSwallow(){
      const TI = TouchInput, O = this._origTI || {};
      if (!O.isTriggered) return;
      TI.isTriggered = O.isTriggered;
      TI.isPressed   = O.isPressed;
      TI.isCancelled = O.isCancelled;
      if (O.isReleased) TI.isReleased = O.isReleased;
      if (O.wheelX) TI.wheelX = O.wheelX;
      if (O.wheelY) TI.wheelY = O.wheelY;
      this._origTI = null; this._swallowTouch=false;
    },

    // Interpreter停止
    _installInterpreterLock(){
      if (this._lockApplied) return;
      this._lockApplied = true;
      const GIp = Game_Interpreter.prototype;
      this._origUpdate = GIp.update;
      const self = this;
      GIp.update = function(){
        if (self.isOpen()) return; // 何もしない＝停止
        return self._origUpdate.apply(this, arguments);
      };
    },
    _uninstallInterpreterLock(){
      if (!this._lockApplied) return;
      Game_Interpreter.prototype.update = this._origUpdate;
      this._origUpdate = null;
      this._lockApplied = false;
    }
  };

  //================ 開く処理 =================
  function openBacklog(force=false){
    if (!force && !allowOpen()) return;
    if (OPEN_MODE==='overlay') Overlay.open();
    else SceneManager.push(Scene_HS_Backlog);
  }

  // プラグインコマンド
  PluginManager.registerCommand(PLUGIN_NAME,'OpenBacklog',()=> openBacklog(false));
  PluginManager.registerCommand(PLUGIN_NAME,'ForceOpenBacklog',()=> openBacklog(true));
  PluginManager.registerCommand(PLUGIN_NAME,'ClearBacklog',()=> $gameSystem.hsClearBacklog());

  // オートオープン：OFF→ON
  if (AUTO_SW>0){
    const _GS_set = Game_Switches.prototype.setValue;
    Game_Switches.prototype.setValue = function(id,val){
      const before = this.value(id);
      _GS_set.apply(this, arguments);
      if (id===AUTO_SW && !before && !!val && allowOpen()) openBacklog(false);
    };
  }

  // ホットキー（任意。既定 none＝フックしない）
  if (HOTKEY && HOTKEY!=='none'){
    const _SB_update = Scene_Base.prototype.update;
    Scene_Base.prototype.update = function(){
      _SB_update.apply(this, arguments);
      try {
        if (!Overlay.isOpen() && allowOpen() && Input.isTriggered(HOTKEY)) {
          openBacklog(false);
          Input.clear();
        }
      } catch(_){}
    };
  }

  // シーン切替時はオーバーレイを閉じる
  const _SM_onSceneCreate = SceneManager.onSceneCreate;
  SceneManager.onSceneCreate = function(){ if (Overlay.isOpen()) Overlay.close(); _SM_onSceneCreate.apply(this, arguments); };

})();

